/**
* \file: partition.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>

#include "model/partition.h"
#include "model/partition_metadata.h"
#include "model/model_internal.h"
#include "app_iface/app_iface_intern.h"

#include "utils/automounter_types_internal.h"

static void partition_signal_state_change(partition_t *partition, partition_state_t old_state);

partition_t *partition_create_new(const char* partition_id,
		device_t* device, partition_metadata_t *metadata)
{
	partition_t *partition;

	partition=malloc(sizeof(partition_t));
	if (partition!=NULL)
	{
		memset(partition, 0, sizeof(partition_t));
		partition->id=strdup(partition_id);
		if (partition->id!=NULL)
		{
			partition->mounted_writable=false;
			partition->state=PARTITION_DETECTED;
			partition->unsupported_reason=_SUPPORTED;
			partition->device=device;
			partition->metadata=metadata;
		}
		else
		{
			partition_free(partition);
			partition=NULL;
		}
	}

	return partition;
}

void partition_free(partition_t *partition)
{
	free(partition->id);

	if (partition->mount_point != NULL)
		free(partition->mount_point);

	partition_metadata_free(partition);

	free(partition);
}

const char *partition_get_id(partition_t *partition)
{
	return partition->id;
}

const char *partition_get_mountsrc(partition_t *partition)
{
	return partition_metadata_get_mountsrc(partition);
}

const char *partition_get_mountfs(partition_t *partition)
{
	return partition_metadata_get_mountfs(partition);
}

const char *partition_get_mountpoint(partition_t *partition)
{
	return partition->mount_point;
}

const char *partition_get_partition_identifier(partition_t *partition)
{
	return partition_metadata_get_partition_identifier(partition);
}

int partition_get_partition_number(partition_t *partition)
{
	return partition_metadata_get_partition_number(partition);
}

bool partition_is_mounted_writable(partition_t *partition)
{
	return partition->mounted_writable;
}

partition_metadata_t *partition_get_metadata(partition_t *partition)
{
	return partition->metadata;
}


device_t *partition_get_device(partition_t *partition)
{
	return partition->device;
}



error_code_t partition_get_error(partition_t *partition)
{
	return partition->error;
}

partition_state_t partition_get_state(partition_t *partition)
{
	return partition->state;
}

const char* partition_get_state_str(partition_state_t state)
{
	return automounter_get_partition_state_string(state);
}

partition_unsupported_reason_t partition_get_unsupported_reason(partition_t *partition)
{
	return partition->unsupported_reason;
}

const char* partition_get_unsupported_reason_str(partition_t *partition)
{
	return automounter_get_unsupported_reason_string(partition->unsupported_reason);
}

void partition_set_state_change_monitor(partition_t *partition, partition_state_change_callback_t callback_func,
					void *callback_data)
{
    partition->state_change_callback_func=callback_func;
    partition->state_change_callback_data=callback_data;  
}

void partition_remove_state_change_monitor(partition_t *partition)
{
    partition->state_change_callback_func=NULL;
    partition->state_change_callback_data=NULL;
}

void partition_set_state_mounting(partition_t *partition, const char* mount_point)
{ 
	partition_state_t old_state;
	old_state=partition->state;	
	if(partition->mount_point!=NULL)
		free(partition->mount_point);
	partition->mount_point=strdup(mount_point);
	partition->state=PARTITION_MOUNTING;
	partition->error=RESULT_OK;
	app_iface_signal_partition_mounting(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_mounted(partition_t *partition, bool writable)
{
  	partition_state_t old_state;
	old_state=partition->state;
	partition->state=PARTITION_MOUNTED;
	partition->mounted_writable=writable;
	partition->error=RESULT_OK;
	app_iface_signal_partition_mounted(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_return_to_state_mounted_on_err(partition_t* partition, error_code_t err)
{
  	partition_state_t old_state;
	old_state=partition->state;
	partition->state=PARTITION_MOUNTED;
	partition->error=err;
	app_iface_signal_partition_mounted(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_unmounting(partition_t *partition)
{
  	partition_state_t old_state;
	old_state=partition->state;
    partition->error=RESULT_OK;
    partition->state=PARTITION_UNMOUNTING;
    app_iface_signal_partition_unmounting(partition);
    partition_signal_state_change(partition,old_state);
}

void partition_set_state_unmounted(partition_t* partition)
{
  	partition_state_t old_state;
	old_state=partition->state;

    partition->error=RESULT_OK;
    partition->state=PARTITION_UNMOUNTED;
    partition->mounted_writable=false;
    app_iface_signal_partition_unmounted(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_remounting(partition_t *partition)
{
  	partition_state_t old_state;
	old_state=partition->state;

    partition->error=RESULT_OK;
    partition->state=PARTITION_REMOUNTING;
    app_iface_signal_partition_remounting(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_invalid(partition_t *partition)
{
  	partition_state_t old_state;
	old_state=partition->state;

	partition->error=RESULT_OK;
	partition->state=PARTITION_INVALID;
	partition->unsupported_reason=_SUPPORTED;
	partition->mounted_writable=false;
	app_iface_signal_partition_invalid(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_unsupported(partition_t* partition, partition_unsupported_reason_t unsupported_reason)
{
  	partition_state_t old_state;
	old_state=partition->state;

    partition->error=RESULT_OK;
    partition->state=PARTITION_UNSUPPORTED;
    partition->unsupported_reason=unsupported_reason;
    app_iface_signal_partition_unsupported(partition);
	partition_signal_state_change(partition,old_state);
}

void partition_set_state_mount_err(partition_t *partition, error_code_t error)
{
  	partition_state_t old_state;
	old_state=partition->state;

      partition->error=error;
      partition->state=PARTITION_MOUNT_ERR;
      app_iface_signal_partition_mount_err(partition);
	partition_signal_state_change(partition,old_state);
}

static void partition_signal_state_change(partition_t *partition, partition_state_t old_state)
{
  if (partition->state_change_callback_func!=NULL)
  {
    partition->state_change_callback_func(partition, old_state,
            partition->state_change_callback_data);
    partition_remove_state_change_monitor(partition);
  }
}
